#include "stdafx.h"
#include <archive\rpanim.h>
#include "keysample.h"
#include "DffExp.h"
#include "matrixmangle.h"
#include "utilities.h"

#include "ikctrl.h"
#ifdef CSSUPPORT
#include "bipexp.h"
#endif

/********************************************
    EMBEDDED ANIM STUFF
**********************************************/

struct TagData
{
        RwInt32 tag;
        RwFrame *frame;
};


RwFrame *
_findTaggedFrame(RwFrame *frame, void *data)
{
    TagData *tagData;

    tagData = (TagData *)data;

    if (_rpAnimSeqGetFrameTag(frame) == tagData->tag)
    {
        tagData->frame = frame;
        return (NULL);
    }

    RwFrameForAllChildren(frame,
                          _findTaggedFrame,
                          data);

    return (frame);
}

void
DFFExport::AddAnimNodeToAnimInfo(INode* node, ANMInfo *ANMData, RwInt32 keyTypes)
{
    int tagID, i;
    BOOL onlyzerokey;

    ANMData->node = node;
    ANMData->keys = (RwBool *) RwMalloc (sizeof(RwBool) * (m_nNumFrames+1));
    if (node->GetUserPropInt("TAG", tagID)) ANMData->tag = tagID;
    else if (node->GetUserPropInt("tag", tagID)) ANMData->tag = tagID;
    else ANMData->tag = -1;

    for (i=0;i<m_nNumFrames+1;i++) ANMData->keys[i] = FALSE;

    /* find it's keyframes */
    FindControlKeys(node->GetTMController(), ANMData->keys, m_tvStart, m_tvEnd, m_noExtraInterpKeys);

    /* If the only key is at zero don't export it */
    onlyzerokey = TRUE;
    for (i=1; i<=m_nNumFrames && onlyzerokey; i++)
    {
        if (ANMData->keys[i]) onlyzerokey = FALSE;
    }
    if (onlyzerokey) ANMData->keys[0] = FALSE;
}

void
DFFExport::AddNodeAndChildsKeys(INode* node, ANMInfo *ANMData, RwInt32 keyTypes, RwInt32 m_tvStart,
                                RwInt32 m_tvEnd, RwInt32 m_nNumFrames)
{
    FindControlKeys(node->GetTMController(), ANMData->keys, m_tvStart, m_tvEnd, m_noExtraInterpKeys);

    /* save the children */
    int nNumChildren = node->NumberOfChildren();
    for (int nChildNum = 0; nChildNum < nNumChildren; nChildNum++)
    {
        INode *childNode = node->GetChildNode(nChildNum);

        AddNodeAndChildsKeys(childNode, ANMData, keyTypes, m_tvStart, m_tvEnd, m_nNumFrames);
    }
}

void
DFFExport::AddAnimNodeToAnimInfoIK(INode* node, ANMInfo *ANMData, RwInt32 keyTypes)
{
    int tagID, i;
    BOOL onlyzerokey;

    ANMData->node = node;
    ANMData->keys = (RwBool *) RwMalloc (sizeof(RwBool) * (m_nNumFrames+1));
    if (node->GetUserPropInt("TAG", tagID)) ANMData->tag = tagID;
    else if (node->GetUserPropInt("tag", tagID)) ANMData->tag = tagID;
    else ANMData->tag = -1;

    for (i=0;i<m_nNumFrames+1;i++) ANMData->keys[i] = FALSE;

    INode *tmpNode = node;
    while (tmpNode && tmpNode->GetParentNode() && !tmpNode->GetParentNode()->IsRootNode())
    {
        tmpNode = tmpNode->GetParentNode();
    }

    /* find it's keyframes */
    AddNodeAndChildsKeys(tmpNode, ANMData, keyTypes, m_tvStart, m_tvEnd, m_nNumFrames);

    /* If the only key is at zero don't export it */
    onlyzerokey = TRUE;
    for (i=1; i<=m_nNumFrames && onlyzerokey; i++)
    {
        if (ANMData->keys[i]) onlyzerokey = FALSE;
    }
    if (onlyzerokey) ANMData->keys[0] = FALSE;
}

#define FRAMESTOSECS(frame) ((float)((frame)*GetTicksPerFrame())/(float)TIME_TICKSPERSEC)

BOOL
DFFExport::AddAnim(INode *node, RpClump *clump, char *name)
{
    TagData tagData;
    RwFrame *frame;
    int tag = -1, tagID;
    int nNumChildren, nChildNum;

    if (!node) return FALSE;

    frame = RpClumpGetFrame(clump);

    if (node->GetUserPropInt("TAG", tagID)) tag = tagID;
    else if (node->GetUserPropInt("tag", tagID)) tag = tagID;

    if (tag != -1 && frame)
    {
        tagData.tag = tag;
        tagData.frame = NULL;

        RwFrameForAllChildren(frame, _findTaggedFrame,
                              (void *)&tagData);

        if (tagData.frame)
        {
            if (m_skinning)
            {
                int j;
                /* look up appropriate bone index */
                for (j=0; j<CSNumBones; j++)
                {
                    if (CSBoneIndexTags[j] == tag)
                    {
                        AddSkinAnimToFrame(node, tagData.frame, name, j);
                    }
                }
            }
            else
            {
                AddAnimToFrame(node, tagData.frame, name);
            }
        }
    }

    /* convert the selected clumps */
    nNumChildren = node->NumberOfChildren();
    for (nChildNum = 0; nChildNum < nNumChildren; nChildNum++)
    {
        INode *childNode = node->GetChildNode(nChildNum);

        AddAnim(childNode, clump, name);
    }

    return TRUE;
}

/* We have a modified function in PowerPipe builds for adding animation
   keyframes for RpSkin. This is because RpSkin requires translation and
   rotational keys to be common and a guaranteed start and end key on all
   bones */
BOOL
DFFExport::AddSkinAnimToFrame(INode *node, RwFrame *frame, char *animName, RwInt32 boneIndex)
{
    ANMInfo ANMData;
    int i;
    RwInt32 numSequences = 0;
    RwInt32 numMatrices;
    RwReal time;
    RwMatrix *mpTransformMatrix;

    if (!node || !frame)
    {
        return FALSE;
    }

    Control *control = node->GetTMController();
    Class_ID controlCID = control->ClassID();

    ANMData.keys = NULL;
    mpTransformMatrix = RwMatrixCreate();

    /* Deal with rotations and translations together for RpSkin */
    /* convert the selected node */
    if (controlCID == IKSLAVE_CLASSID)
    {
        AddAnimNodeToAnimInfoIK(node, &ANMData, KEYTYPE_ROTATION | KEYTYPE_TRANSLATION);
    }
    else
    {
        AddAnimNodeToAnimInfo(node, &ANMData, KEYTYPE_ROTATION | KEYTYPE_TRANSLATION);
    }

    if (m_filterTopLevelCSKeys)
    {
        AddTopLevelBipedPartKeys(rootNode, &ANMData, m_tvStart, m_tvEnd, m_nNumFrames);
    }

    numMatrices = 0;
    if (ANMData.keys)
    {
        ANMData.keys[0] = TRUE;
        ANMData.keys[m_nNumFrames] = TRUE;
        for (i=0; i <= m_nNumFrames; i++)
        {
            if(ANMData.keys[i]) numMatrices++;
        }
    }
    else
    {
        numMatrices = 0;
    }
    if (numMatrices > 1)
    {        
        /* Log the number of frames created. */
        char temp[256];
        sprintf(temp,"Skin animation %s created %i unified rotational and translational keys",
            animName,numMatrices);
        m_WarningsList.add(CWarning(wtInformational,node->GetName(),temp));
        m_WarningsList.m_RotKeys+=numMatrices;
        m_WarningsList.m_TransKeys+=numMatrices;

        for (i=0; i <= m_nNumFrames; i++)
        {
            if(ANMData.keys[i])
            {
                AnimKeyFrameListEntry *newEntry = NULL;
                AnimKeyFrameListEntry *temp = NULL;

                RtQuat quat;

                Matrix3 objMat = GetObjMatrix(ANMData.node, (i*GetTicksPerFrame())+m_tvStart, FALSE);

                /* if root node get difference from starting position */
                if (node->GetParentNode()->IsRootNode())
                {
                    Matrix3 objMatStart = GetObjMatrix(ANMData.node, m_tvStart, FALSE);
                    objMat *= Inverse(objMatStart);
                }

                //Character studio has a different idea of what is "up" than the rest of MAX.
                //Reorient this matrix to have the same "up" as everything else
                objMat = ToRenderwareAxisSystem( objMat, node );

                //let's face it, you can NOT represent scaling - uniform, non-uniform, negative,
                //or otherwise, as a unit quaternion, so remove it.
                //Besides, RtQuatConvertFromMatrix will assert
                objMat = Uniform_Matrix( objMat );

                Matrix3ToRwMatrix(objMat, mpTransformMatrix);

                RtQuatConvertFromMatrix(&quat, mpTransformMatrix);

                time = FRAMESTOSECS(i);

                /* Create a new key in the SKA key list */
                newEntry = (AnimKeyFrameListEntry *)RwMalloc(sizeof(AnimKeyFrameListEntry));
                temp = &SKAKeyFrameList;
                newEntry->time = FRAMESTOSECS(i);
                while(temp->next && temp->next->time < newEntry->time) temp = temp->next;
                newEntry->next = temp->next;
                temp->next = newEntry;
                newEntry->bone = boneIndex;                
                newEntry->q.imag = quat.imag;
                newEntry->q.real = quat.real;
                newEntry->t = mpTransformMatrix->pos;

                /* check that this is either not a subhierarchy export or that this frame is 
                   in the subhierarchy */
                if ((!m_HAnimSubHierarchyAnim) ||
                    (node == m_selectedNodeInSelectedHierarchy) ||
                    (NodeHasAncestor(node, m_selectedNodeInSelectedHierarchy)))
                {
                    /* Create a new key in the HAnim key list */
                    newEntry = (AnimKeyFrameListEntry *)RwMalloc(sizeof(AnimKeyFrameListEntry));
                    temp = &AnimKeyFrameList;
                    newEntry->time = FRAMESTOSECS(i);
                    while(temp->next && temp->next->time < newEntry->time) temp = temp->next;
                    newEntry->next = temp->next;
                    temp->next = newEntry;
                    newEntry->bone = boneIndex;                
                    newEntry->q.imag = quat.imag;
                    newEntry->q.real = quat.real;
                    newEntry->t = mpTransformMatrix->pos;
                }
            }
        }
    }

    /* Better RwFree some memory I guess */
    RwFree(ANMData.keys);

    RwMatrixDestroy(mpTransformMatrix);

    return TRUE;
}

BOOL
DFFExport::AddAnimToFrame(INode *node, RwFrame *frame, char *animName)
{
    ANMInfo ANMData;
    int i;
    RwInt32 numSequences = 0;
    RwInt32 numMatrices;
    RwReal time;
    RwMatrix *mpTransformMatrix;

    if (!node || !frame)
    {
        return FALSE;
    }

    Control *control = node->GetTMController();
    Class_ID controlCID = control->ClassID();

    ANMData.keys = NULL;
    mpTransformMatrix = RwMatrixCreate();

    /* First deal with rotations */
    /* convert the selected node */
    if (controlCID == IKSLAVE_CLASSID)
    {
        AddAnimNodeToAnimInfoIK(node, &ANMData, KEYTYPE_ROTATION);
    }
    else
    {
        AddAnimNodeToAnimInfo(node, &ANMData, KEYTYPE_ROTATION);
    }

    numMatrices = 0;
    if (ANMData.keys)
    {
        for (i=0; i <= m_nNumFrames; i++)
        {
            if(ANMData.keys[i]) numMatrices++;
        }
    }
    else
    {
        numMatrices = 0;
    }
    if (numMatrices > 1)
    {
        RpAnimSequence  *rotSeq;
        RwInt32 matIndex;
        RwReal prevTime = 0.0f;

        /* Create an animation sequence */
        rotSeq = RpAnimSequenceCreate(animName, rpROTATE,
                                      numMatrices, numMatrices - 1);

        /* Add the sequence to the frame */
        RpAnimFrameAddSequence(frame, rotSeq);

        /* Make it so the frame owns it */
        RpAnimSequenceDestroy(rotSeq);

        matIndex = 0;
        for (i=0; i <= m_nNumFrames; i++)
        {
            if(ANMData.keys[i])
            {

                RtQuat quat;

                Matrix3 objMat = GetObjMatrix(ANMData.node, (i*GetTicksPerFrame())+m_tvStart, FALSE);

                /* if root node get difference from starting position */
                if (node->GetParentNode()->IsRootNode())
                {
                    Matrix3 objMatStart = GetObjMatrix(ANMData.node, m_tvStart, FALSE);
                    objMat *= Inverse(objMatStart);
                }

                //Character studio has a different idea of what is "up" than the rest of MAX.
                //Reorient this matrix to have the same "up" as everything else
                objMat = ToRenderwareAxisSystem( objMat, node );
                
                //let's face it, you can NOT represent scaling - uniform, non-uniform, negative,
                //or otherwise, as a unit quaternion, so remove it.
                //Besides, RtQuatConvertFromMatrix will assert
                objMat = Uniform_Matrix( objMat );

                Matrix3ToRwMatrix(objMat, mpTransformMatrix);

                RtQuatConvertFromMatrix(&quat, mpTransformMatrix);
                RpAnimSequenceSetRotateKey(rotSeq, matIndex, &quat);

                time = FRAMESTOSECS(i);

                /* Set up the interpolators */
                if (matIndex > 0)
                {
                    /* Set the interpolators for the animation sequence */
                    RpAnimSequenceSetInterpolator(rotSeq, matIndex - 1, matIndex - 1,
                                                  matIndex, time - prevTime);
                }

                matIndex++;
                prevTime = time;
            }
        }
        /* Log the creation of the animation sequence. */
        char temp[256];
        sprintf(temp,"Animation %s created %i rotational keys",
            animName,numMatrices);
        m_WarningsList.add(CWarning(wtInformational,node->GetName(),temp));
        m_WarningsList.m_RotKeys+=numMatrices;
    }

    /* Better RwFree some memory I guess */
    RwFree(ANMData.keys);

    /* Now do the translation */
    /* convert the selected node */
    if (controlCID == IKSLAVE_CLASSID)
    {
        AddAnimNodeToAnimInfoIK(node, &ANMData, KEYTYPE_TRANSLATION);
    }
    else
    {
        AddAnimNodeToAnimInfo(node, &ANMData, KEYTYPE_TRANSLATION);
    }

    numMatrices = 0;
    if (ANMData.keys)
    {
        for (i=0; i <= m_nNumFrames; i++)
        {
            if(ANMData.keys[i]) numMatrices++;
        }
    }
    else
    {
        numMatrices = 0;
    }
    if (numMatrices > 1)
    {
        RpAnimSequence  *transSeq;
        RwInt32 matIndex;
        RwReal prevTime = 0.0f;

        /* Create an animation sequence */
        transSeq = RpAnimSequenceCreate(animName, rpTRANSLATE,
                                        numMatrices, numMatrices - 1);

        /* Add the sequence to the frame */
        RpAnimFrameAddSequence(frame, transSeq);

        /* Make it so the frame owns it */
        RpAnimSequenceDestroy(transSeq);

        matIndex = 0;
        for (i=0; i <= m_nNumFrames; i++)
        {
            if(ANMData.keys[i])
            {

                Matrix3 objMat = GetObjMatrix(ANMData.node, (i*GetTicksPerFrame())+m_tvStart, FALSE);

                /* if root node get difference from starting position */
                if (node->GetParentNode()->IsRootNode())
                {
                    Matrix3 objMatStart = GetObjMatrix(ANMData.node, m_tvStart, FALSE);
                    objMat *= Inverse(objMatStart);
                }

                //Character studio has a different idea of what is "up" than the rest of MAX.
                //Reorient this matrix to have the same "up" as everything else
                objMat = ToRenderwareAxisSystem( objMat, node );

                Matrix3ToRwMatrix(objMat, mpTransformMatrix);

                RpAnimSequenceSetTranslateKey(transSeq, matIndex, &mpTransformMatrix->pos);

                time = FRAMESTOSECS(i);

                /* Set up the interpolators */
                if (matIndex > 0)
                {
                    /* Set the interpolators for the animation sequence */
                    RpAnimSequenceSetInterpolator(transSeq, matIndex - 1, matIndex - 1,
                                                  matIndex, time - prevTime);
                }

                matIndex++;
                prevTime = time;
            }
        }
        /* Log the creation of the animation sequence. */
        char temp[256];
        sprintf(temp,"Animation %s created %i translational keys",
            animName,numMatrices);
        m_WarningsList.add(CWarning(wtInformational,node->GetName(),temp));
        m_WarningsList.m_TransKeys+=numMatrices;
    }

    /* Better RwFree some memory I guess */
    RwFree(ANMData.keys);

    RwMatrixDestroy(mpTransformMatrix);

    return TRUE;
}

/*********************************************************

   Code to add anim sequences to the clump since we
   created them

 *********************************************************/

static RpAnimSequence *
_rpAnimClumpAddSequenceCallback(RpAnimSequence *animSequence, void *data)
{
    RpAnimClumpAddSequence((RpClump *)data,
                           RpAnimSequenceGetName(animSequence),
                           NULL,
                           NULL);

    return (animSequence);
}

static RwFrame *
_rpAnimClumpAddAllSequencesCallBack(RwFrame *frame, void *data)
{
    RpAnimFrameForAllSequences(frame,
                               _rpAnimClumpAddSequenceCallback,
                               data);

    RwFrameForAllChildren(frame,
                          _rpAnimClumpAddAllSequencesCallBack,
                          data);

    return (frame);
}

RpClump *
_rpAnimClumpForAllFramesAddSequences(RpClump *clump)
{
    if (clump)
    {
        RwFrame            *clumpFrame = RpClumpGetFrame(clump);

        if (clumpFrame)
        {
            _rpAnimClumpAddAllSequencesCallBack(clumpFrame, (void *) clump);
        }

        return (clump);
    }

    return (NULL);
}

/**************************************************************************************************

  Routine to query keys from any type of max controller. For standard types of controllers we
  simply ask for keys, for complex types we don't know how to deal with we output a key for
  every frame.
  Where we can find the keys if there are multiple keys in our range or keys span the required range
  we also add in keys at the start and end 

  ************************************************************************************************/
#define TICKSTOFRAMES(ticks) ((int)ceil((ticks) / (GetTicksPerFrame())))

/* Reactor controller class id's */
#define REACTORFLOAT 0x717d7d1f
#define REACTORPOS 0x7ac5cae4
#define REACTORP3 0x19080908
#define REACTORROT 0x2a8734eb
#define REACTORSCALE 0x13c4451c
#define REACTORFLOAT_CLASS_ID	Class_ID(REACTORFLOAT, 0x124c173b)
#define REACTORPOS_CLASS_ID		Class_ID(REACTORPOS, 0x904a56b3)
#define REACTORP3_CLASS_ID		Class_ID(REACTORP3, 0x3b617839)
#define REACTORROT_CLASS_ID		Class_ID(REACTORROT, 0x57f47da6)
#define REACTORSCALE_CLASS_ID	Class_ID(REACTORSCALE, 0x2ccb3388)

/* list controllers */
#define FLOATLIST_CONTROL_CLASS_ID		0x4b4b1000
#define POINT3LIST_CONTROL_CLASS_ID		0x4b4b1001
#define POSLIST_CONTROL_CLASS_ID		0x4b4b1002
#define ROTLIST_CONTROL_CLASS_ID		0x4b4b1003
#define SCALELIST_CONTROL_CLASS_ID		0x4b4b1004

#define DYNAMICSPOS_CONTROL_ID Class_ID(0x69403713, 0x5ea269cd)
#define DYNAMICSROT_CONTROL_ID Class_ID(0x4bff6bd7, 0x199b5c5a)

void
FindControlKeys(Control *control, RwBool *keys, RwInt32 animStart, RwInt32 animEnd, RwBool noExtraKeys)
{
    RwInt32 i;
    RwInt32 startKey = TICKSTOFRAMES(animStart);
    
    if (control)
    {        
        Class_ID cid = control->ClassID();
        if (control->ClassID() == Class_ID(PRS_CONTROL_CLASS_ID, 0) ||
            control->ClassID() == Class_ID(LOOKAT_CONTROL_CLASS_ID, 0) ||
            control->ClassID() == IKSLAVE_CLASSID)
        {
            /* for PRS controllers we process all the sub controls */                        
            FindControlKeys(control->GetPositionController(), keys, animStart, animEnd, noExtraKeys);
            FindControlKeys(control->GetRotationController(), keys, animStart, animEnd, noExtraKeys);
            FindControlKeys(control->GetRollController(), keys, animStart, animEnd, noExtraKeys);
        }
#ifdef CSSUPPORT
        else if (control->ClassID() == BIPBODY_CONTROL_CLASS_ID)
        {  
            for (int s=0;s<3;s++)
            {
                Animatable *sub = control->SubAnim(s);
      
                if (sub)
                {
                    /* Do biped stuff */
                    Tab<TimeValue> times;                    
                    int first;

                    first = sub->GetKeyTimes(times, FOREVER, 0);

                    if (times.Count() > 1 &&
                        times[first] <= animEnd &&
                        times[times.Count()-1] >= animStart)
                    {
                        keys[TICKSTOFRAMES(animStart)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(animEnd)-startKey] = TRUE;
                    }

                    for (i = first; i < times.Count(); i++)
                    {                
                        if (times[i] >= animStart &&                           
                            times[i] <= animEnd)
                        {
                            keys[TICKSTOFRAMES(times[i])-startKey] = TRUE;
                        }
                    }
                }
            }
        }
#endif /* CSSUPPORT */
        else if (control->ClassID() == Class_ID(EXPR_POS_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(EXPR_P3_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(EXPR_FLOAT_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(EXPR_SCALE_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(EXPR_ROT_CONTROL_CLASS_ID, 0))
        {
            /* for expression controllers we output a key at every frame */
            for (i=TICKSTOFRAMES(animStart); i<=TICKSTOFRAMES(animEnd); i++)
            {
                keys[i-startKey] = TRUE;
            }
        }        
        else if (control->ClassID() == REACTORFLOAT_CLASS_ID ||
                 control->ClassID() == REACTORPOS_CLASS_ID ||
                 control->ClassID() == REACTORP3_CLASS_ID ||
                 control->ClassID() == REACTORROT_CLASS_ID ||
                 control->ClassID() == REACTORSCALE_CLASS_ID)
        {
            /* for reactor controllers we output a key at every frame (for now) */
            for (i=TICKSTOFRAMES(animStart); i<=TICKSTOFRAMES(animEnd); i++)
            {
                keys[i-startKey] = TRUE;
            }
        }
        else if (control->ClassID() == Class_ID(PATH_CONTROL_CLASS_ID, 0))
        {
            /* for path controllers we output a key at every frame (for now since 
               we don't have enough spline control) */
            for (i=TICKSTOFRAMES(animStart); i<=TICKSTOFRAMES(animEnd); i++)
            {
                keys[i-startKey] = TRUE;
            }
        }        
        else if (control->ClassID() == Class_ID(FLOATLIST_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(POINT3LIST_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(POSLIST_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(ROTLIST_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(SCALELIST_CONTROL_CLASS_ID, 0))
        {
            /* for list (block) controllers we process all the sub controls */
            int i;
            int numSubs = control->NumSubs();
            for (i=0; i<numSubs; i++)
            {
                Control *sub = (Control *)control->SubAnim(i);
                FindControlKeys(sub, keys, animStart, animEnd, noExtraKeys);
            }  
        }        
        else if (control->ClassID() == DYNAMICSPOS_CONTROL_ID ||
                 control->ClassID() == DYNAMICSROT_CONTROL_ID)
        {
            /* for dynamics controllers we output a key at every frame */
            for (i=TICKSTOFRAMES(animStart); i<=TICKSTOFRAMES(animEnd); i++)
            {
                keys[i-startKey] = TRUE;
            }
        }        
        else if (control->ClassID() == Class_ID(EULER_CONTROL_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(LOCAL_EULER_CONTROL_CLASS_ID, 0))
        {
            int i;
            for (i=0; i<3; i++)
            {
                Control *sub = (Control *)control->SubAnim(i);
                FindControlKeys(sub, keys, animStart, animEnd, noExtraKeys);
            }        
        }
        else if (control->ClassID() == Class_ID(TCBINTERP_FLOAT_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(TCBINTERP_POSITION_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(TCBINTERP_POINT3_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(TCBINTERP_SCALE_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(HYBRIDINTERP_FLOAT_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(HYBRIDINTERP_POSITION_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(HYBRIDINTERP_POINT3_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(HYBRIDINTERP_SCALE_CLASS_ID, 0) ||
                 control->ClassID() == Class_ID(HYBRIDINTERP_COLOR_CLASS_ID, 0))
        {
            /* TCB interpolated keys. For all of these we will output the actual
               keys and keys placed 5% and 15% into the gaps either side of the 
               actual keys, unless m_NoExtraInterpKeys is set. These extra keys
               should simulate the smooth curves allowed by TCB controllers */ 
            Tab<TimeValue> times;            
            int first;
            int prevKey = animStart;

            first = control->GetKeyTimes(times, FOREVER, 0);

            /* Remove insertion of start and end keys to make animation run to the correct length
			if (times.Count() > 1 &&
                times[first] <= animEnd &&
                times[times.Count()-1] >= animStart)
            {
                // add a start and end key
                int i = 0;
                int s = animStart, e = animEnd;
                while (i < times.Count() && times[i] < animStart) i++;
                if (i == times.Count() || times[i] != animStart)
                {
                    times.Insert(i, 1, &s);
                }
                i = 0;
                while (i < times.Count() && times[i] < animEnd) i++;
                if (i == times.Count() || times[i] != animEnd)
                {
                    times.Insert(i, 1, &e);
                }
            }*/

            for (i = first; i < times.Count(); i++)
            {               
                int time = times[i];
                if (time >= animStart &&                           
                        time <= animEnd)
                {
                    keys[TICKSTOFRAMES(time)-startKey] = TRUE;
                    if ((time != animStart)&&(!noExtraKeys))
                    {
                        /* if not the first key add keys 5% and 15% after
                        prev key and 5% and 15% before this key */
                        float fivepercent = (float)(time - prevKey) * 0.05f;
                        float fifteenpercent = (float)(time - prevKey) * 0.15f;

                        keys[TICKSTOFRAMES(prevKey+fivepercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(prevKey+fifteenpercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(time-fivepercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(time-fifteenpercent)-startKey] = TRUE;

                        prevKey = time;
                    }
                }                
            }
        }
        /* for the rotation controllers we need to check for rotations >= 180
           degrees and add extra keys, we'll stick in extras at start then 
           start + (x * 90) until we hit the end key */
        else if (control->ClassID() == Class_ID(LININTERP_ROTATION_CLASS_ID, 0))
        {
            Tab<TimeValue> times;            
            int first;
            IKeyControl *keycontrol = GetKeyControlInterface(control);
            
            /* first add keys the normal way */
            first = control->GetKeyTimes(times, FOREVER, 0);

            if (times.Count() > 1 &&
                times[first] <= animEnd &&
                times[times.Count()-1] >= animStart)
            {
                keys[TICKSTOFRAMES(animStart)-startKey] = TRUE;
                keys[TICKSTOFRAMES(animEnd)-startKey] = TRUE;
            }

            for (i = first; i < times.Count(); i++)
            {               
                int time = times[i];
                if (times[i] >= animStart &&                           
                    times[i] <= animEnd)
                {
                    keys[TICKSTOFRAMES(times[i])-startKey] = TRUE;
                }
            }

            /* now loop through the actual keys adding any extras */
            for(i=0; i < keycontrol->GetNumKeys()-1; i++)
            {   
                ILinRotKey thekey1, thekey2;
                AngAxis rot1, rot2;
                float diff;                
                keycontrol->GetKey(i, &thekey1);
                keycontrol->GetKey(i+1, &thekey2);
                
                rot1 = AngAxis(thekey1.val);
                rot2 = AngAxis(thekey2.val);

                diff = rot2.angle - rot1.angle;
                while (diff > HALFPI)
                {
                    int time;
                    diff -= HALFPI;

                    time = (int)(((rot2.angle - diff - rot1.angle) / (rot2.angle - rot1.angle)) 
                            * ((float)thekey2.time - (float)thekey1.time)) 
                            + thekey1.time;
                    if (time >= animStart && time <= animEnd)
                    {
                        keys[TICKSTOFRAMES(time)-startKey] = TRUE;
                    }
                }                
            }
        }
        else if (control->ClassID() == Class_ID(HYBRIDINTERP_ROTATION_CLASS_ID, 0))
        {
            /* TCB interpolated keys. For all of these we will output the actual
               keys and keys placed 5% and 15% into the gaps either side of the 
               actual keys unless m_NoExtraInterpKeys is set. These extra keys should
               simulate the smooth curves allowed by TCB controllers */ 
            Tab<TimeValue> times;            
            int first;
            int prevKey = animStart;
            IKeyControl *keycontrol = GetKeyControlInterface(control);

            first = control->GetKeyTimes(times, FOREVER, 0);

            /* Remove insertion of start and end keys to make animation run to the correct length
			if (times.Count() > 1 &&
                times[first] <= animEnd &&
                times[times.Count()-1] >= animStart)
            {
                // add a start and end key
                int i = 0;
                int s = animStart, e = animEnd;
                while (i < times.Count() && times[i] < animStart) i++;
                if (i == times.Count() || times[i] != animStart)
                {
                    times.Insert(i, 1, &s);
                }
                i = 0;
                while (i < times.Count() && times[i] < animEnd) i++;
                if (i == times.Count() || times[i] != animEnd)
                {
                    times.Insert(i, 1, &e);
                }
            }*/
            
            for (i = first; i < times.Count(); i++)
            {               
                int time = times[i];
                if (time >= animStart &&                           
                    time <= animEnd)
                {
                    keys[TICKSTOFRAMES(time)] = TRUE;
                    if ((time != animStart)&&(!noExtraKeys))
                    {
                        /* if not the first key add keys 5% and 15% after
                        prev key and 5% and 15% before this key */
                        float fivepercent = (float)(time - prevKey) * 0.05f;
                        float fifteenpercent = (float)(time - prevKey) * 0.15f;

                        keys[TICKSTOFRAMES(prevKey+fivepercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(prevKey+fifteenpercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(time-fivepercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(time-fifteenpercent)-startKey] = TRUE;

                        prevKey = time;
                    }
                }                
            }

            /* now loop through the actual keys adding any extras */
            for(i=0; i < keycontrol->GetNumKeys()-1; i++)
            {   
                IBezQuatKey thekey1, thekey2;
                AngAxis rot1, rot2;
                float diff;                
                keycontrol->GetKey(i, &thekey1);
                keycontrol->GetKey(i+1, &thekey2);
                
                rot1 = AngAxis(thekey1.val);
                rot2 = AngAxis(thekey2.val);

                diff = rot2.angle - rot1.angle;
                while (diff > HALFPI)
                {
                    int time;
                    diff -= HALFPI;

                    time = (int)(((rot2.angle - diff - rot1.angle) / (rot2.angle - rot1.angle)) 
                            * ((float)thekey2.time - (float)thekey1.time)) 
                            + thekey1.time;
                    if (time >= animStart && time <= animEnd)
                    {
                        keys[TICKSTOFRAMES(time)-startKey] = TRUE;
                    }
                }                
            }

        }
        else if (control->ClassID() == Class_ID(TCBINTERP_ROTATION_CLASS_ID, 0))
        {
            /* TCB interpolated keys. For all of these we will output the actual
               keys and keys placed 5% and 15% into the gaps either side of the 
               actual keys unless m_NoExtraInterpKeys is set. These extra keys should
               simulate the smooth curves allowed by TCB controllers */ 
            Tab<TimeValue> times;            
            int first;
            int prevKey = animStart;
            IKeyControl *keycontrol = GetKeyControlInterface(control);

            first = control->GetKeyTimes(times, FOREVER, 0);

            /* Remove insertion of start and end keys to make animation run to the correct length
			if (times.Count() > 1 &&
                times[first] <= animEnd &&
                times[times.Count()-1] >= animStart)
            {
                // add a start and end key
                int i = 0;
                int s = animStart, e = animEnd;
                while (i < times.Count() && times[i] < animStart) i++;
                if (i == times.Count() || times[i] != animStart)
                {
                    times.Insert(i, 1, &s);
                }
                i = 0;
                while (i < times.Count() && times[i] < animEnd) i++;
                if (i == times.Count() || times[i] != animEnd)
                {
                    times.Insert(i, 1, &e);
                }
            }*/
            
            for (i = first; i < times.Count(); i++)
            {               
                int time = times[i];
                if (time >= animStart &&                           
                    time <= animEnd)
                {
                    keys[TICKSTOFRAMES(time)-startKey] = TRUE;
                    if ((time != animStart)&&(!noExtraKeys))
                    {
                        /* if not the first key add keys 5% and 15% after
                        prev key and 5% and 15% before this key */
                        float fivepercent = (float)(time - prevKey) * 0.05f;
                        float fifteenpercent = (float)(time - prevKey) * 0.15f;

                        keys[TICKSTOFRAMES(prevKey+fivepercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(prevKey+fifteenpercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(time-fivepercent)-startKey] = TRUE;
                        keys[TICKSTOFRAMES(time-fifteenpercent)-startKey] = TRUE;

                        prevKey = time;
                    }
                }                
            }

            /* now loop through the actual keys adding any extras */
            for(i=0; i < keycontrol->GetNumKeys()-1; i++)
            {   
                ITCBRotKey thekey1, thekey2;
                float diff;                
                keycontrol->GetKey(i, &thekey1);
                keycontrol->GetKey(i+1, &thekey2);
                
                diff = thekey2.val.angle - thekey1.val.angle;
                while (diff > HALFPI)
                {
                    int time;
                    diff -= HALFPI;

                    time = (int)(((thekey2.val.angle - diff - thekey1.val.angle) / (thekey2.val.angle - thekey1.val.angle)) 
                            * ((float)thekey2.time - (float)thekey1.time)) 
                            + thekey1.time;
                    if (time >= animStart && time <= animEnd)
                    {
                        keys[TICKSTOFRAMES(time)-startKey] = TRUE;
                    }
                }                
            }

        }
        else
        {
            /* default types including BIPSLAVE_CONTROL_CLASS_ID & FOOTPRINT_CLASS_ID */ 
            Tab<TimeValue> times;            
            int first;

            first = control->GetKeyTimes(times, FOREVER, 0);

            if (times.Count() > 1 &&
                times[first] <= animEnd &&
                times[times.Count()-1] >= animStart)
            {
                keys[TICKSTOFRAMES(animStart)-startKey] = TRUE;
                keys[TICKSTOFRAMES(animEnd)-startKey] = TRUE;
            }

            for (i = first; i < times.Count(); i++)
            {               
                int time = times[i];
                if (times[i] >= animStart &&                           
                    times[i] <= animEnd)
                {
                    keys[TICKSTOFRAMES(times[i])-startKey] = TRUE;
                }
            }
        }
        
    }
}

void 
DFFExport::DestroyKeyFrameList(AnimKeyFrameListEntry *head)
{
    while(head->next)
    {
        AnimKeyFrameListEntry *next = head->next->next;
        RwFree(head->next);
        head->next = next;
    }
}

